<?php

namespace cnb\admin;

// don't load directly
defined( 'ABSPATH' ) || die( '-1' );

use cnb\admin\api\CnbAppRemote;
use cnb\admin\api\CnbAppRemotePayment;
use cnb\admin\domain\CnbDomainController;
use cnb\admin\models\CnbUser;
use cnb\admin\profile\CnbProfileController;
use WP_Error;

class CnbAdminAjax {
    /**
     * part of domain-upgrade
     *
     * @return void
     */
    public function domain_upgrade_get_checkout() {
        do_action( 'cnb_init', __METHOD__ );
        $planId   = filter_input( INPUT_POST, 'planId', FILTER_SANITIZE_STRING );
        $domainId = filter_input( INPUT_POST, 'domainId', FILTER_SANITIZE_STRING );

        $url             = admin_url( 'admin.php' );
        $redirect_link   =
            add_query_arg(
                array(
                    'page'    => 'call-now-button-domains',
                    'action'  => 'upgrade',
                    'id'      => $domainId,
                    'upgrade' => 'success'
                ),
                $url );
        $callbackUri     = esc_url_raw( $redirect_link );
        $checkoutSession = CnbAppRemotePayment::cnb_remote_post_subscription( $planId, $domainId, $callbackUri );

        if ( is_wp_error( $checkoutSession ) ) {
            $custom_message_data = $checkoutSession->get_error_data( 'CNB_ERROR' );
            if ( ! empty( $custom_message_data ) ) {
                $custom_message_obj = json_decode( $custom_message_data );
                $message            = $custom_message_obj->message;
                // Strip "request_id"
                if ( stripos( $message, '; request-id' ) !== 0 ) {
                    $message = preg_replace( '/; request-id.*/i', '', $message );
                }
                // Replace "customer" with "domain"
                $message = str_replace( 'customer', 'domain', $message );
                wp_send_json( array(
                    'status'  => 'error',
                    'message' => $message
                ) );
            } else {
                wp_send_json( array(
                    'status'  => 'error',
                    'message' => $checkoutSession->get_error_message()
                ) );
            }
        } else {
            // Get link based on Stripe checkoutSessionId
            wp_send_json( array(
                'status'  => 'success',
                'url' => $checkoutSession->url
            ) );
        }
        do_action( 'cnb_finish' );
        wp_die();
    }

    /**
     * called via jQuery.post
     * @return void
     */
    public function settings_profile_save() {
        do_action( 'cnb_init', __METHOD__ );
        $data = array();
        // Security note: the nonce will be checked via update_user (below),
        // and we sanitize the data via filter_var below
        // phpcs:ignore WordPress.Security
        wp_parse_str( $_POST['data'], $data );
        $controller = new CnbProfileController();
        $nonce      = filter_var( $data['_wpnonce'], FILTER_SANITIZE_STRING );
        $profile    = filter_var( $data['user'], FILTER_SANITIZE_STRING,
            FILTER_REQUIRE_ARRAY | FILTER_FLAG_NO_ENCODE_QUOTES );
        $user       = CnbUser::fromObject( $profile );

        $result = $controller->update_user( $nonce, $user );
        wp_send_json( $result );
        do_action( 'cnb_finish' );
        wp_die();
    }

    public function cnb_email_activation() {
        do_action( 'cnb_init', __METHOD__ );
        $admin_url = esc_url( admin_url( 'admin.php' ) );

        $custom_email = trim( filter_input( INPUT_POST, 'admin_email', FILTER_SANITIZE_STRING ) );
        if ( is_email( $custom_email ) ) {
            $data = CnbAppRemote::cnb_remote_email_activation( $custom_email, $admin_url );
        } else {
            $data = new WP_Error( 'CNB_EMAIL_INVALID', __( 'Please enter a valid e-mail address.' ) );
            if ( empty( $custom_email ) ) {
                $data = new WP_Error( 'CNB_EMAIL_EMPTY', __( 'Please enter a valid e-mail address.' ) );
            }
        }
        wp_send_json( $data );
        do_action( 'cnb_finish' );
        wp_die();
    }

    private static function cnb_time_format_( $time ) {
        $time_format    = get_option( 'time_format' );
        $time_formatted = strtotime( $time );

        return date_i18n( $time_format, $time_formatted );
    }

    public function time_format() {
        do_action( 'cnb_init', __METHOD__ );
        $start = trim( filter_input( INPUT_POST, 'start', FILTER_SANITIZE_STRING ) );
        $stop  = trim( filter_input( INPUT_POST, 'stop', FILTER_SANITIZE_STRING ) );
        wp_send_json( array(
                'start' => self::cnb_time_format_( $start ),
                'stop'  => self::cnb_time_format_( $stop ),
            )
        );
        do_action( 'cnb_finish' );
        wp_die();
    }

    public function get_plans() {
        do_action( 'cnb_init', __METHOD__ );
        $domain_controller    = new CnbDomainController();

        $plans                = CnbAppRemotePayment::cnb_remote_get_plans();

        // Hardcoded fallback values in case the API call fails
        $eur_yearly_per_month = 4.16;
        $eur_discount = 17;
        $usd_yearly_per_month = 4.99;
        $usd_discount = 17;

        if (is_array($plans)) {
            $eur_yearly_plan      = array_filter( $plans, function ( $plan ) {
                return $plan->nickname === 'powered-by-eur-yearly';
            } );
            $eur_yearly_plan      = array_pop( $eur_yearly_plan );
            $eur_yearly_per_month = round( $eur_yearly_plan->price / 12.0, 2 );

            $usd_yearly_plan      = array_filter( $plans, function ( $plan ) {
                return $plan->nickname === 'powered-by-usd-yearly';
            } );
            $usd_yearly_plan      = array_pop( $usd_yearly_plan );
            $usd_yearly_per_month = round( $usd_yearly_plan->price / 12.0, 2 );

            $eur_monthly_plan = array_filter( $plans, function ( $plan ) {
                return $plan->nickname === 'powered-by-eur-monthly';
            } );
            $usd_monthly_plan = array_filter( $plans, function ( $plan ) {
                return $plan->nickname === 'powered-by-usd-monthly';
            } );

            // Calculate discounts
            $eur_discount = $domain_controller->get_discount_percentage( $eur_yearly_plan, array_pop( $eur_monthly_plan ) );
            $usd_discount = $domain_controller->get_discount_percentage( $usd_yearly_plan, array_pop( $usd_monthly_plan ) );
        }

        wp_send_json( array(
            'eur_per_month' => $eur_yearly_per_month,
            'eur_discount'  => $eur_discount,
            'usd_per_month' => $usd_yearly_per_month,
            'usd_discount'  => $usd_discount,
        ) );
        do_action( 'cnb_finish' );
        wp_die();
    }
}
